{ A form containing a simple graph and a stringgrid, used to display the model
  output without having to switch to another application. The output data is
  stored in the stringgrid when the form is shown. If DisplayStyle is dstable
  then the Stringgrid is shown. If DisplayStyle is dsChart then the stringgrid
  is hidden and the chart and a list box with the variable names is shown. Note
  that the data is always stored in the stringgrid even if it's the chart that
  is being displayed. This is because the chart reads the series from the
  stringgrid.
  }
unit display;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  TeEngine, Series, DBChart, StdCtrls, Grids, ExtCtrls, TeeProcs, Chart,
  Db, DBTables, Menus, DBGrids, stypes, Printers, Mask;

const
  Maxseries = 10;
  FirstRow = 3;

type
  plotarray=array[1..MaxSeries] of string;
  TFmDisplayData = class(TForm)
    DlgPrint: TPrintDialog;
    DisplayMenu: TMainMenu;
    MIWindow: TMenuItem;
    MICloseDisplay: TMenuItem;
    MIChart: TMenuItem;
    MIXaxis: TMenuItem;
    MIYaxis: TMenuItem;
    MIUpdateChart: TMenuItem;
    MIDefineX: TMenuItem;
    MIScaleX: TMenuItem;
    MIDefineY: TMenuItem;
    MIScaleY: TMenuItem;
    PnBottom: TPanel;
    SgModelOutput: TStringGrid;
    MINewFile: TMenuItem;
    PnBottomLeft: TPanel;
    LbxSeries: TListBox;
    PnBottomRight: TPanel;
    RgValueType: TRadioGroup;
    BtnUpdateChart: TButton;
    BtnClearSeries: TButton;
    BtnCloseDisplay: TButton;
    LbDirections: TLabel;
    MIPrintChart: TMenuItem;
    PnTop: TPanel;
    ChOutput: TChart;
    Series1: TPointSeries;
    Series2: TPointSeries;
    Series3: TPointSeries;
    Series4: TPointSeries;
    Series5: TPointSeries;
    Series6: TPointSeries;
    Series7: TPointSeries;
    Series8: TPointSeries;
    Series9: TPointSeries;
    Series10: TPointSeries;
    pnCalibrate: TPanel;
    lblPar1: TLabel;
    lblPar2: TLabel;
    lblPar3: TLabel;
    lblPar4: TLabel;
    BtnRunCalibrate: TButton;
    cbParams: TCheckBox;
    cbState: TCheckBox;
    mePar1: TMaskEdit;
    mePar2: TMaskEdit;
    mePar3: TMaskEdit;
    mePar4: TMaskEdit;
    function columnmax(ycolumn:integer):double; 
    function columnmin(ycolumn:integer):double;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure BtnUpdateChartClick(Sender: TObject);
    procedure BtnCloseDisplayClick(Sender: TObject);
    procedure MIPrintDisplayClick(Sender: TObject);
    procedure ChooseSeries(Sender: TObject);
    procedure BtnClearSeriesClick(Sender: TObject);
    procedure MINewFileClick(Sender: TObject);
    procedure FillListBox(ListBox:TListBox);
    procedure ScaleClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure MIPrintChartClick(Sender: TObject);
    procedure BtnRunCalibrateClick(Sender: TObject);
    procedure meParExit(Sender: TObject);
    procedure lblParClick(sender: Tobject);
    procedure refreshMeParCalibrate;
    procedure cbParamsClick(Sender: TObject);
    procedure cbStateClick(Sender: TObject);
  private
    { Private declarations }
    FFilename:String;
    FCurrentAxis:TAxis;
    FxAxis:string;
    FDisplayStep:double;
    FDisplayStyle:TDisplayStyle;
    FNumberofSeries:integer;
    FSeriestoPlot:plotarray;
    FNeedtoReadData:Boolean;
    FParameterChosen: array[1..4] of Boolean;
    procedure SetDisplayStep(stepvalue:double);
    procedure ClearGrid;
    function GetxAxis:string;
    procedure SetxAxis(SeriesName:string);
    function GetColumnNumber(seriesname:string):integer;
  public
    { Public declarations }
    autoshowchart: Boolean;
    property Filename:String read FFilename write FFilename;
    property CurrentAxis:TAxis read FCurrentAxis write FCurrentAxis;
    property displaystep:double read FDisplayStep write SetDisplayStep;
    property xAxis:string read GetxAxis write SetxAxis;
    property DisplayStyle:TDisplayStyle read FDisplayStyle write FDisplayStyle;
    property NumberofSeries:integer read FNumberofSeries write FNumberofSeries;
    property NeedtoReadData:Boolean read FNeedtoReadData write FNeedtoReadData;
    procedure AddSeriestoPlot(seriesname:string);
    procedure RemoveSeriestoPlot(seriesname:string);
    procedure SetAxisScale(axis:TAxis; min, max, increm:double; auto:Boolean);
    procedure SetAxisType(axis:TAxis; axistype:TAxisType);
    procedure SetSeriestoPlot(Listbox:TListBox);
    procedure ClearSeriestoPlot(Listbox:TListBox);
    procedure UpdateChart;
    procedure ResizeListbox(listbox:Tlistbox);
    procedure UpdateStringGrid;
end;

var
  FmDisplayData: TFmDisplayData;

implementation

uses fileio, frontend, ScaleDlg, data, SeriesForm, ReloadDlg, ParamList,
     calculate, Options, math;

{$R *.DFM}

// Procedure to create the form, set up the stringgrid, and set up the listbox.
procedure TFmDisplayData.FormCreate(Sender: TObject);
var
 i : integer;
begin
  chOutput.Title.Text.text := modeldef.modelname; // Chart title
  with SgModelOutput do // Set up the Stringgrid
   begin
    colcount := modeldef.numdrive + modeldef.numprocess + 1;
    // don't need numstate in the above count because the derivatives are no
    // longer in the table so the count of the derivatives in numprocess takes
    // care of numstate.
    for i := 0 to colcount do cells[i,0] := inttostr(i);
    cells[0,1] := 'Time';             // Column name
    cells[0,2] := ModelDef.timeunit;   // Column units
    for i := 1 to modeldef.numdrive do         // Add drivers to stringgrid
     begin
      cells[i,1] := drive[i].name;     // Column name
      cells[i,2] := drive[i].units;    // Column units
     end;
    for i := 1 to modeldef.numstate do         // Add state variables to grid
     begin
      cells[modeldef.numdrive + i,1] := stat[i].name;   // Column name
      cells[modeldef.numdrive + i,2] := stat[i].units;  // Column units
     end;
    for i := ModelDef.numstate + 1 to modeldef.numprocess do       // Add process variables to grid
     begin
      cells[modeldef.numdrive + i,1] := proc[i].name;  // Column name
      cells[modeldef.numdrive + i,2] := proc[i].units; // Column units
     end;
   end;

 for i := 1 to 4 do
   FParameterChosen[i] := False;

 FillListBox(LbxSeries);
 DisplayStep := FmOptions.RunOptions.Time_step;
 xAxis := 'Time';
 NeedtoReadData := True;
 autoshowchart := false;
end;



{ This procedure reads in the driver variables, state variables and process
  variables from the driver file and the output file. The values are stored in
  the stringgrid. If DisplayStyle is dsChart the stringgrid is hidden and the
  chart is shown and if DisplayStyle is dsTable the chart is hidden and the
  stringgrid is shown.}
procedure TFmDisplayData.FormShow(Sender: TObject);
var
 BtnSpacing: integer;
begin
 if NeedtoReadData then UpdateStringGrid;
 // Change properties of objects on the form depending on whether form will show
 // a chart or a table
 FmDisplayData.FormResize(MainForm);
 if DisplayStyle = dsChart then
  begin
   // Objects which are not visible are also disabled so that they can't be
   // accidently tabbed to.
   SgModelOutput.Enabled := False;  // Disable grid
   SgModelOutput.Visible := False;  // Hide grid
   ChOutput.Enabled := True;        // Enable chart
   ChOutput.Visible := True;        // Show chart
   MIChart.Enabled := True;         // Enable chart menu
   MIChart.Visible := True;         // Show chart menu
   PnBottom.Height := FmDisplayData.Height div 3; // Change panel bottom panel height so that listbox is visible
   LbxSeries.Visible := True;       // Show Listbox
   LbxSeries.Enabled := True;       // Enable Listbox
   LbDirections.Visible := True;    // Show label with listbox directions
   RgValueType.Visible := True;
   RgValueType.Enabled := True;
   BtnSpacing := ((PnBottom.Height div 2) div 3);
   BtnUpdateChart.Top := (PnBottom.Height div 2) +
                         (BtnSpacing - BtnUpdateChart.Height) div 2 - 1;
   BtnClearSeries.Top := (PnBottom.Height div 2) + BtnSpacing +
                         (BtnSpacing - BtnClearSeries.Height) div 2 - 1;
   BtnCloseDisplay.Top := (PnBottom.Height div 2) + 2*BtnSpacing +
                         (BtnSpacing - BtnCloseDisplay.Height) div 2 - 1;
                        // Move Close Display button below Update chart button
   BtnUpdateChart.Enabled := True;   // Enable Update chart button
   PnTop.Show;
  end
 else
  begin
   SgModelOutput.Enabled := True;   // Enable grid
   SgModelOutput.Visible := True;   // Show grid
   ChOutput.Enabled := False;       // Disable chart
   ChOutput.Visible := False;       // Hide chart
   MIChart.Visible := False;        // Hide chart menu
   MIChart.Enabled := False;        // Disable chart menu
   PnBottom.Height := FmDisplayData.Height div 20; // Shrink bottom panel to allow stringgrid more room to show data
   if PnBottom.Height < 30 then PnBottom.Height := 30;
   LbxSeries.Visible := False;      // Hide listbox
   LbxSeries.Enabled := False;      // Disable listbox
   LbDirections.Visible := False;   // Hide listbox directions
   RgValueType.Visible := False;
   RgValueType.Enabled := False;
   BtnCloseDisplay.Top := (PnBottom.Height - BtnCloseDisplay.Height) div 2 - 1;
         // Move Close Display button to top of panel
   BtnUpdateChart.Enabled := False;  // Disable Update chart button
   PnTop.Hide;
  end;
 RgValueType.Height := (PnBottomRight.Height div 2) - RgValueType.Top;
end;

{ When the update chart button is clicked, clear any old data on the graph,
  then plot series in seriestoplot variable. }
procedure TFmDisplayData.BtnUpdateChartClick(Sender: TObject);
begin
 UpdateChart;
 refreshMeParCalibrate;
end;

function TFmDisplayData.columnmax(ycolumn:integer):double;
var
  Ymax:double;
  j:integer;
begin
   Ymax:=strtofloat(SgModelOutput.cells[ycolumn,FirstRow]);
   for j:=FirstRow to SgModelOutput.RowCount - 1 do
        Ymax:=max(Ymax,strtofloat(SgModelOutput.cells[ycolumn,j]));
   columnmax:=Ymax;
end;

function TFmDisplayData.columnmin(ycolumn:integer):double;
var
  Ymin:double;
  j:integer;
begin
   Ymin:=strtofloat(SgModelOutput.cells[ycolumn,FirstRow]);
   for j:=FirstRow to SgModelOutput.RowCount - 1 do
        Ymin:=min(Ymin,strtofloat(SgModelOutput.cells[ycolumn,j]));
   columnmin:=Ymin;
end;

procedure TFmDisplayData.UpdateChart;
var
 i,j,xcolumn,ycolumn:integer;
 Ymax,Ymin,Mmax,Mmin,dum:double;
begin
 SetSeriestoPlot(LbxSeries);
 for i := 1 to MaxSeries do    // Clear and inactivate all the series
  begin
   choutput.series[i-1].clear;
   Choutput.series[i-1].active := false;
  end;
  // Put title on Bottom Axis
 xcolumn := GetColumnNumber(xAxis);
 ChOutput.BottomAxis.Title.Caption := SgModelOutput.cells[xcolumn,1];
 try
   // Activate the number of series to plot.
  for i := 0 to NumberofSeries - 1 do Choutput.series[i].active := true;
  //***************************

   ycolumn := GetColumnNumber(FSeriestoPlot[1]);
   Ymax:=columnmax(ycolumn);
   Ymin:=columnmin(ycolumn);
   for j := 1 to NumberofSeries - 1 do
   begin
      ycolumn := GetColumnNumber(FSeriestoPlot[j+1]);
      Ymax:=max(Ymax,columnmax(ycolumn));
      Ymin:=min(Ymin,columnmin(ycolumn));
   end;
   Mmax:=1;
   Mmin:=1;
   While abs(Ymax)>10 do
      begin
         Ymax:=Ymax/10;
         Mmax:=Mmax*10;
      end;
   While (abs(Ymax)<1) and (abs(Ymax)>0) do
      begin
         Ymax:=Ymax*10;
         Mmax:=Mmax/10;
      end;

   While abs(Ymin)>10 do
      begin
         Ymin:=Ymin/10;
         Mmin:=Mmin*10;
      end;
   While (abs(Ymin)<1) and (abs(Ymin)>0) do
      begin
         Ymin:=Ymin*10;
         Mmin:=Mmin/10;
      end;
  if Mmax>Mmin then
     begin
        Ymin:=Ymin*Mmin/Mmax;
        Mmin:=Mmax;
     end
  else
     begin
        Ymax:=Ymax*Mmax/Mmin;
        Mmax:=Mmin;
     end;
  //***************************
  { Add each series to the chart. The values plotted depend on whether
  actual, relative or cummulative change is checked in the radio group. }
  for j := 0 to NumberofSeries - 1 do
   begin
    ycolumn := GetColumnNumber(FSeriestoPlot[j+1]);
    for i := FirstRow to SgModelOutput.RowCount - 1 do  // loop over rows in the grid
     begin   // Add series to chart
      if RgValueType.ItemIndex = 1 then   // Plot relative change of series
       begin
         Choutput.Series[j].addxy(strtofloat(SgModelOutput.Cells[xcolumn,i]),
            strtofloat(SgModelOutput.Cells[ycolumn,i])/
            strtofloat(SgModelOutput.Cells[ycolumn,FirstRow]),
            SgModelOutput.cells[xcolumn,i],clTeeColor);
       end
      else if RgValueType.ItemIndex = 2 then  // Plot Cummulative change
       begin
         Choutput.Series[j].addxy(strtofloat(SgModelOutput.Cells[xcolumn,i]),
            strtofloat(SgModelOutput.Cells[ycolumn,i]) -
            strtofloat(SgModelOutput.Cells[ycolumn,FirstRow]),
            SgModelOutput.cells[xcolumn,i],clTeeColor);
       end
      else // Value type is "Actual Values" so plot the values in the stringgrid
       begin
         Choutput.Series[j].addxy(strtofloat(SgModelOutput.Cells[xcolumn,i]),
            strtofloat(SgModelOutput.Cells[ycolumn,i])/Mmax,
            SgModelOutput.cells[xcolumn,i],clTeeColor);
       end;
      ChOutput.Series[j].title := SgModelOutput.cells[ycolumn,1];
     end;
    end;
 except
  on Exception do MessageDlg('Cannot update chart. Invalid series.',
                                  mtInformation, [mbOK], 0);
 end;

 if RgValueType.ItemIndex = 0 then
  begin
   if (Ymin>0) and (ymin < ymax/3) then
    ymin := 0
   else
      begin
         if (Ymax<0) and (ymax>ymin/3) then Ymax:=0
         else
           begin
             dum := ymax - ymin;
             ymax := ymax + dum/10;
             ymin := ymin - dum/10;
           end;
      end;
   ChOutput.LeftAxis.Title.Caption := 'x 10^ ' + floattostr(log10(Mmax));
   SetAxisScale(axLeft,ymin,Ymax,0,false);
  end
 else
  begin
   SetAxisScale(axLeft,0,0,0,true);
   ChOutput.LeftAxis.Title.Caption := ' ';
  end;

 SetAxisScale(axBottom,0,0,0,true);

end;

procedure TFmDisplayData.BtnCloseDisplayClick(Sender: TObject);
begin
 NeedtoReadData := False;  // Stringgrid contains data. Don't need to update grid
                      // unless there's a new run or parameters changed.
 Close;
end;

{ Prints the visible portion of the Display Form only. Useful mostly for
  printing chart.}
procedure TFmDisplayData.MIPrintDisplayClick(Sender: TObject);
begin
 if DlgPrint.execute then
     Print;
end;

{ Choose the series for independent and dependent axis' of the chart. }
procedure TFmDisplayData.ChooseSeries(Sender: TObject);
begin
 UpdateChart;
 if Sender = MIDefineX then
  CurrentAxis := axBottom
 else
  CurrentAxis := axLeft;
 FmSeries.ShowModal; // Show the FmSeries which contains a listbox with all
                     // the series names.
end;

// Clear old data from the StringGrid to prevent data overlap when new data is added.
procedure TFmDisplayData.ClearGrid;
var
 i,j:integer;
begin
 for i := FirstRow to SgModelOutput.RowCount do
   for j := 0 to SgModelOutput.ColCount - 1 do
    SgModelOutput.Cells[j,i] := ''; // Set all cells to empty strings
 SgModelOutput.RowCount := FirstRow; // Decrease size of grid
 NeedtoReadData := True; // Data needs to be read from the output file
end;

procedure TFmDisplayData.BtnClearSeriesClick(Sender: TObject);
begin
  ClearSeriestoPlot(LbxSeries);
end;

procedure TFmDisplayData.MINewFileClick(Sender: TObject);
begin
  MainForm.MIViewOutFileClick(FmDisplayData);
  FormShow(DisplayMenu);
  BtnUpdateChartClick(Displaymenu);
end;

procedure TFmDisplayData.FillListBox(ListBox:TListBox);
var                      // fix this so that it loops of the columns in the string grid. that way there's no mistake in the column headings.
 i:integer;
begin
 ListBox.Items.Clear;
 ListBox.Items.Add('Time');
 for i := 1 to ModelDef.numdrive do   // Add driver names to listbox
  ListBox.Items.Add(drive[i].name);
 for i := 1 to ModelDef.numstate do   // Add state variable names to listbox
  ListBox.Items.Add(stat[i].name);
 for i := ModelDef.numstate + 1 to ModelDef.numprocess do // Add process variable names to listbox
  ListBox.Items.Add(proc[i].name);
end;

procedure TFmDisplayData.SetDisplayStep(stepvalue:double);
begin
{  if stepvalue < FmOptions.RunOptions.Time_step then
   begin    // Warn the user and correct the problem.
     MessageDlg('Display Step must be greater than or equal to the Time Step.'
                 + ' Display step set to current Time step.',
                 mtWarning, [mbOK], 0);
     stepvalue := FmOptions.RunOptions.time_step;  // Set the display step to the default value.
   end
  // If the DisplayStep is not a multiple of the Time step
  else if abs(frac(stepvalue/FmOptions.RunOptions.Time_Step))>0.001 then
   begin   // Warn the user and correct the problem.
     MessageDlg('Display Step must be a multiple of Time Step.',
                 mtWarning, [mbOK], 0);
     stepvalue := FmOptions.RunOptions.Time_Step; // Set the display step to the default value.
   end;            }
  FDisplayStep := stepvalue;
  NeedtoReadData := True; // Tell the display form to reread in the output data.
end;

procedure TFmDisplayData.SetxAxis(SeriesName:string);
begin
 FxAxis := SeriesName;
end;

function TFmDisplayData.GetxAxis:string;
begin
 GetxAxis := FxAxis;
end;

procedure TFmDisplayData.AddSeriestoPlot(seriesname:string);
var
 index:integer;
begin
 if NumberofSeries = MaxSeries then
  MessageDlg('Cannot display series. Maximum is 10 series.',mtError,[mbOK],0)
 else
  begin
    index := LbxSeries.Items.IndexOf(seriesname);
    NumberofSeries := NumberofSeries + 1;
    LbxSeries.Selected[index] := True;
    FSeriestoPlot[NumberofSeries] := seriesname;
  end;
end;

procedure TFmDisplayData.RemoveSeriestoPlot(seriesname:string);
var
 temparray:plotarray;
 i,index:integer;
begin
  index := 1;
  temparray := FSeriestoPlot;
  for i := 1 to NumberofSeries do
   begin
     if temparray[i] = seriesname then
       index := i;
   end;
  for i := index to NumberofSeries - 1 do
   begin
     FSeriestoPlot[i] := temparray[i + 1];
   end;
  NumberofSeries := NumberofSeries - 1;
end;

procedure TFmDisplayData.SetSeriestoPlot(Listbox:TListBox);
var
  i:integer;
begin
  NumberofSeries := 0;
  for i := 0 to Listbox.Items.Count - 1 do
    begin
     if Listbox.Selected[i] then
      begin
       AddSeriestoPlot(Listbox.Items[i]);
      end;
    end;
end;

procedure TFmDisplayData.ClearSeriestoPlot(Listbox:TListBox);
var
  i:integer;
begin
  for i := 0 to Listbox.Items.Count - 1 do
    begin
     if Listbox.Selected[i] then
      begin
       RemoveSeriestoPlot(Listbox.Items[i]);
       Listbox.Selected[i] := False;
      end;
    end;
end;

procedure TFmDisplayData.SetAxisType(axis:TAxis; axistype:TAxisType);
begin
 if axis = axBottom then
  begin
    if axistype = tyLinear then
      ChOutput.BottomAxis.Logarithmic := False
    else
      ChOutput.BottomAxis.Logarithmic := True;
  end
 else
  begin
    if axistype = tyLinear then
      ChOutput.LeftAxis.Logarithmic := False
    else
      ChOutput.LeftAxis.Logarithmic := True;
  end;
end;

function TFmDisplayData.GetColumnNumber(seriesname:string):integer;
var
 j,num:integer;
begin
 num := -1;
 for j := 0 to SgModelOutput.Colcount do
  begin
   if SgModelOutput.Cells[j,1] = seriesname then num := j;
  end;
 GetColumnNumber := num;
end;

// Show a dialog to modify the axis scale
procedure TFmDisplayData.ScaleClick(Sender: TObject);
begin
// UpdateChart;
 if Sender = MIScaleX then
  CurrentAxis := axBottom
 else
  CurrentAxis := axLeft;

 DlgScale.ShowModal;  // Show the Scale Change dialog box
end;

procedure TFmDisplayData.SetAxisScale(axis:TAxis; min, max, increm:double;
                                        auto:Boolean);
var
 minvalue,maxvalue:double;
begin
 if auto then
   begin
     if axis = axBottom then
       begin
         ChOutput.BottomAxis.Automatic := True;
       end
     else
       begin
         ChOutput.LeftAxis.Automatic := True;
         minvalue := ChOutput.MinYValue(ChOutput.LeftAxis);
         maxvalue := ChOutput.MaxYValue(ChOutput.LeftAxis);
         if minvalue > 0 then minvalue := 0;
         ChOutput.LeftAxis.SetMinMax(minvalue,maxvalue);
       end;
   end
 else
   begin
     if axis = axBottom then
       begin
         ChOutput.BottomAxis.Automatic := False;
         ChOutput.BottomAxis.Increment := Increm;
         ChOutput.BottomAxis.SetMinMax(min,max);
       end
     else
       begin
         ChOutput.LeftAxis.Automatic := False;
         ChOutput.LeftAxis.Increment := Increm;
         ChOutput.LeftAxis.SetMinMax(min,max);
       end;
   end;
end;

procedure TFmDisplayData.FormResize(Sender: TObject);
begin
 ResizeListbox(LbxSeries);
end;

procedure TFmDisplayData.ResizeListbox(listbox:Tlistbox);
var
 numcol:double;
begin
 if listbox.name = 'LbxSeries' then
  begin
   numcol := (FmDisplayData.ClientWidth-PnBottomRight.Width)/200.0;
      //        (listbox.Width/listbox.Columns); //
   if numcol < 1 then numcol := 1;
   listbox.columns := round(numcol)
  end
 else
  begin
   numcol := (FmSeries.ClientWidth)/200.0;//(listbox.Width/listbox.Columns); //
   if numcol < 1 then numcol := 1;
   listbox.columns := round(numcol)
  end;
end;

procedure TFmDisplayData.MIPrintChartClick(Sender: TObject);
{var
  rect:Trect;  }
begin
{ if MainForm.DlgPrint.execute then
   begin
     Printer.Orientation := poLandscape;
     Rect.Top := 5;
     Rect.Left := 5;
     Rect.Right := Printer.Pagewidth - 100;
     Rect.Bottom := Printer.Pageheight - 100;
     Printer.BeginDoc;
     try
      ChOutput.Draw(Printer.Canvas, Rect);
      Printer.EndDoc;
      Printer.Orientation := poPortrait;
     except
      Printer.Abort;
      raise;
     end;
   end;   }
  if MessageDlg('Print chart to the default printer?', mtInformation,
              [mbOK, mbCancel], 0) = mrOK then ChOutput.PrintLandscape;
end;

procedure TFmDisplayData.UpdateStringGrid;
var
 rownumber,j:integer;
 CurrentTime:double;
 tempstate:statearray;
 tempproc:processarray;
 tempdrive:drivearray;
begin
 ClearGrid;
 FmDisplayData.Caption := 'Model Output - File: ' + FmDisplayData.Filename;
 rownumber := FirstRow;  // Start at first empty row in the stringgrid.
 { Copy state, driver and process variables into temporary variables. Use the
   temporary variables for reading in the values from the files so that the
   current state of the modeled system isn't changed should a read error occur.}
 tempstate := stat;
 tempproc := proc;
 tempdrive := drive;
 // Open output file for reading.
 OpenOutputFile(Filename,ModelDef.numstate,tempstate,ModelDef.numprocess,
          tempproc,flread);
 { The code below is nested in a try:finally loop so that if an error occurs
    the output file will be closed. }
 try
      // Read a line of the output file
 while ReadOutputFile(CurrentTime, ModelDef.numdrive, tempdrive,
          ModelDef.numstate, tempstate, ModelDef.numprocess, tempproc)
      // While you have not reached the end of the output file begin
 do begin
{ // If CurrentTime is an even multiple of DisplayStep then write values to grid
 writetogrid := MainForm.xMultipleofY(CurrentTime,DisplayStep);
 {CurrentTime/DisplayStep - round(CurrentTime/DisplayStep);}
// if{ (writetogrid) and }(CurrentTime >= Time_Start) then
  begin  // Write to grid    // First increase total number of rows
    SgModelOutput.rowcount := SgModelOutput.rowcount + 1;
    SgModelOutput.Cells[0,rownumber] := format('%g',[CurrentTime]); // Write time
    for j:= 1 to ModelDef.numdrive    // Write drivers
     do SgModelOutput.Cells[j,rownumber] := format('%.6g',[tempdrive[j].value]);

    for j:= 1 to ModelDef.numstate    // Write state variables
     do SgModelOutput.Cells[ModelDef.numdrive + j,rownumber]
                := format('%.6g',[tempstate[j].value]);
    for j:= ModelDef.numstate + 1 to ModelDef.numprocess  // Write process variables
     do SgModelOutput.Cells[ModelDef.numdrive + j, rownumber]
                := format('%.6g',[tempproc[j].value]);
    rownumber := rownumber + 1;      // Increment row number writing to
  end;
 end;
 finally
  CloseOutputFile;
 end;
end;

procedure TFmDisplayData.BtnRunCalibrateClick(Sender: TObject);
begin
 if MessageDlg('Rerun model using current run options?', mtConfirmation, [mbYes,mbNo], 0) = mrYes then
  begin
   dlgReload.okBtn.click;
   mainForm.btnRun.click;
   updateStringGrid;
  end;
end;

procedure TFmDisplayData.refreshMeParCalibrate;
var
 thisME : tMaskEdit;
 thisLabel : tLabel;
 i, j : integer;
 parIndex : integer;
begin
 with fmDisplayData {pnCalibrate} do   // fix how can you change the owner to pnCalibrate
  begin
   for i := 1 to componentCount-1 do //look at all the MaskEdits
    begin
     if components[i] is TMaskEdit then
      begin
       thisME := components[i] as TMaskEdit;
       for j := 1 to componentCount-1 do //look at all the Labels
        begin
         if components[j] is TLabel then
          begin
           thisLabel := components[j] as TLabel;
           if thisLabel.Tag = thisMe.Tag then // if the label and the ME are together
            begin
             if pos('Parameter', thisLabel.Caption) = 0 then
              begin
               parIndex :=
                fmCalculate.getArrayIndex(vtParameter, thisLabel.Caption);
               thisME.Text := floatToStr(par[parIndex].value);
              end;
            end;
          end;
        end;
      end;
    end;
  end;
end;

procedure TFmDisplayData.meParExit(Sender: TObject);
var
 thisME : tMaskEdit;
 thisLabel : tLabel;
 i : integer;
 parIndex : integer;
begin
 thisME := sender as TMaskEdit;
 with fmDisplayData {pnCalibrate} do   // fix how can you change the owner to pnCalibrate
  begin
   for i := 1 to componentCount-1 do
    begin
     if components[i] is TLabel then
      begin
       thisLabel := components[i] as TLabel;
       if thisLabel.Tag = thisMe.Tag then
        begin
         parIndex := fmCalculate.getArrayIndex(vtParameter, thisLabel.Caption);
         try
          if thisME.modified then
           par[parIndex].value := strToFloat(thisME.text);
         except
          messageDlg('Please choose a number', mtWarning, [mbOK], 0);
          fmDisplayData.ActiveControl := thisME;
         end;
        end;
      end;
    end;
  end;
 RefreshMeParCalibrate;
end;

procedure TFmDisplayData.lblParClick(sender: Tobject);
var
 tempCaption : string;
 tempName : string;
begin
// Getting the number of the label, i.e. 1, 2, 3, or 4 so fmParamList knows which label/maskedit to modify
 tempName := (sender as tControl).name;
 delete(tempName, 1, 6);
 fmParamList.whichParSelection := strToInt(tempName);

 tempCaption := (sender as tLabel).caption;
 // highlight the parameter currently displayed
 if FParameterChosen[strtoint(tempName)] then
  begin
   fmParamList.lbParamSymbols.ItemIndex :=
     fmCalculate.getArrayIndex(vtParameter, tempCaption)-1;
   fmParamList.lbParamNames.ItemIndex := fmParamList.lbParamSymbols.ItemIndex;
  end;

 fmParamList.showmodal;
end;

procedure TFmDisplayData.cbParamsClick(Sender: TObject);
begin
 if cbParams.Checked  = true
 then DlgReload.cbParams.Checked := true
 else DlgReload.cbParams.Checked := false;
end;

procedure TFmDisplayData.cbStateClick(Sender: TObject);
begin
 if cbState.Checked  = true
 then DlgReload.cbState.Checked := true
 else DlgReload.cbState.Checked := false;
end;


end.


